import React, { useState, useEffect, useRef } from 'react';
import { NavLink, useLocation } from 'react-router-dom';
import Navigation from './nav/Navigation.tsx';
import Menu from '@mui/material/Menu';
import MenuItem from '@mui/material/MenuItem';
import ExpandMoreIcon from '@mui/icons-material/ExpandMore';
import ExpandLessIcon from '@mui/icons-material/ExpandLess';
 
const API_URL = import.meta.env.VITE_API_URL;

interface HeaderProperties {
    facebook: string;
    twitter: string;
    instagram: string;
    youtube: string;
    tiktok: string;
    pintrest: string;
    about: string;
    address: string;
    phone: string;
    email: string;
    copy: string;
    whatsapp: string;
    whatsappMessage: string;
}

interface MenuItemType {
    id: number;
    title: string;
    path: string;
    activePaths?: string[];
    subMenu?: MenuItemType[];
}

const Header = ({ LOGO, apiUrl }: { LOGO: string; apiUrl: string }) => {
    const [properties, setProperties] = useState<HeaderProperties | null>(null);
    const [menuItems, setMenuItems] = useState<MenuItemType[]>([]);
    const [loading, setLoading] = useState(true);
    const [isSticky, setIsSticky] = useState(false);
    const language = 'en';
    const location = useLocation();
    const [isMenuOpen, setIsMenuOpen] = useState(false);
    const [anchorEl, setAnchorEl] = useState<null | HTMLElement>(null);
    const [openMenuId, setOpenMenuId] = useState<number | null>(null);

    const isHomePage =
        location.pathname === '/' ||
        location.pathname === `/${language}` ||
        location.pathname === `/${language}/`;

    const menuRef = useRef<HTMLUListElement>(null);
    const toggleMenu = () => {
        setIsMenuOpen((prevState) => !prevState);
        document.body.classList.toggle('fix', !isMenuOpen);
    };

    const hidePopup = () => {
        setIsMenuOpen(false);
        document.body.classList.remove('fix');
    };
    useEffect(() => {
        const fetchData = async () => {
            try {
                const [propsRes, menuRes] = await Promise.all([
                    fetch(apiUrl),
                    fetch(`${API_URL}menu/lang/${language}`),
                ]);
                if (!propsRes.ok || !menuRes.ok) throw new Error('Failed to fetch');

                const propsData = await propsRes.json();
                const menuData = await menuRes.json();

                setProperties(propsData);
                setMenuItems(menuData);
                setLoading(false);
            } catch (e) {
                console.error(e);
                setLoading(false);
            }
        };
        fetchData();

        const onScroll = () => {
            if (isHomePage) setIsSticky(window.scrollY > 80);
        };
        onScroll();
        window.addEventListener('scroll', onScroll);
        return () => window.removeEventListener('scroll', onScroll);
    }, [apiUrl, isHomePage]);

    useEffect(() => {
        const handleClickOutside = (event: MouseEvent) => {
            if (
                menuRef.current &&
                !menuRef.current.contains(event.target as Node)
            ) {
                setAnchorEl(null);
                setOpenMenuId(null);
            }
        };
        document.addEventListener('mousedown', handleClickOutside);
        return () => document.removeEventListener('mousedown', handleClickOutside);
    }, []);

    if (loading || !properties) return null;

    const handleMenuOpen = (event: React.MouseEvent<HTMLElement>, id: number) => {
        if (openMenuId === id) {
            setAnchorEl(null);
            setOpenMenuId(null);
        } else {
            setAnchorEl(event.currentTarget);
            setOpenMenuId(id);
        }
    };

    const handleMenuClose = () => {
        setAnchorEl(null);
        setOpenMenuId(null);
    };

    return (
        <header
            id="header"
            className={`header ${isSticky ? 'sticky' : ''} ${isHomePage ? 'no-inner inner-home' : 'inner-page sticky'
                } clearfix`}
        >
            <div className="header the-new-header">
                <div className="container11 auto-container111 pl15 pr15">
                    <div className="row item-list-header">
                        <div className="col col-sm-12 col-md-12 header-left">
                            <NavLink to="/" className="logo">
                                <img
                                    width={710}
                                    height={96}
                                    src={!isSticky && isHomePage ? LOGO : LOGO}
                                    className="attachment-full size-full"
                                    alt=""
                                />
                            </NavLink>

                            <ul className="items hide-on-mobile" ref={menuRef}>
                                {menuItems.map(({ id, title, path, activePaths, subMenu }) => {
                                    const isOpen = openMenuId === id;

                                    return (
                                        <li key={id}>
                                            {subMenu ? (
                                                <>
                                                    {/** Manually check if button should be "active" */}
                                                    {(() => {
                                                        const isActiveButton =
                                                            activePaths &&
                                                            activePaths.some((p) => location.pathname.startsWith(p));

                                                        return (
                                                            <button
                                                                type="button"
                                                                aria-controls={isOpen ? `submenu-${id}` : undefined}
                                                                aria-haspopup="true"
                                                                aria-expanded={isOpen ? 'true' : undefined}
                                                                onClick={(e) => handleMenuOpen(e, id)}
                                                                className={`menu-button${isActiveButton ? ' active' : ''}`}
                                                                style={{
                                                                    background: 'none',
                                                                    border: 'none',
                                                                    cursor: 'pointer',
                                                                    font: 'inherit',
                                                                    display: 'flex',
                                                                    alignItems: 'center',
                                                                    gap: 4,
                                                                }}
                                                            >
                                                                {title}
                                                                {isOpen ? (
                                                                    <ExpandLessIcon fontSize="small" />
                                                                ) : (
                                                                    <ExpandMoreIcon fontSize="small" />
                                                                )}
                                                                
                                                            </button>
                                                            
                                                        );
                                                    })()}
                                           

                                                    <Menu
                                                        id={`submenu-${id}`}
                                                        anchorEl={anchorEl}
                                                        open={isOpen}
                                                        onClose={handleMenuClose}
                                                        anchorOrigin={{
                                                            vertical: 'bottom',
                                                            horizontal: 'left',
                                                        }}
                                                        transformOrigin={{
                                                            vertical: 'top',
                                                            horizontal: 'left',
                                                        }}
                                                        PaperProps={{
                                                            sx: {
                                                                mt: 2, // adds margin-top to push submenu down by 8px (1 spacing unit)
                                                            },
                                                          }}
                                                        MenuListProps={{
                                                            'aria-labelledby': `submenu-button-${id}`,
                                                        }}
                                                    >
                                                        {subMenu.map((sub) => (
                                                            <MenuItem
                                                                key={sub.id}
                                                                component={NavLink}
                                                                to={sub.path}
                                                                onClick={handleMenuClose}
                                                                selected={
                                                                    location.pathname === sub.path ||
                                                                    (sub.activePaths &&
                                                                        sub.activePaths.some((p) =>
                                                                            location.pathname.startsWith(p)
                                                                        ))
                                                                }
                                                            >
                                                                {sub.title}
                                                            </MenuItem>
                                                        ))}
                                                    </Menu>
                                                </>
                                            ) : (
                                                <NavLink
                                                    to={path}
                                                    className={({ isActive }) => {
                                                        if (isActive) return 'active';
                                                        if (
                                                            activePaths &&
                                                            activePaths.some((p) =>
                                                                location.pathname.startsWith(p)
                                                            )
                                                        )
                                                            return 'active';
                                                        return '';
                                                    }}
                                                >
                                                    {title}
                                                </NavLink>
                                            )}
                                        </li>
                                    );
                                })}
                            </ul>

                            <div className="right-element">
                             
                                <div className="header__content">
                                    <div className="whatsapp-burger">
                                        <NavLink
                                            to={`https://wa.me/${properties.whatsapp.replace(
                                                '+',
                                                ''
                                            )}?text=${encodeURIComponent(properties.whatsappMessage)}`}
                                            className={({ isActive }) => (isActive ? 'active' : '')}
                                        >
                                            <img
                                                src="/assets/images/whatsapp.png"
                                                alt="WhatsApp Us"
                                            />
                                            
                                        </NavLink>
                                    </div>
                                    <div className="header__menu">
                                        <div id="google_translate_element"></div>
                                        <div
                                            className="header__menu-burger btn-menu"
                                            onClick={toggleMenu}
                                            data-burger-menu=""
                                        >
                                            <div className="header__menu-burger-container">
                                                <div className="header__menu-burger">
                                                    <svg
                                                        xmlns="http://www.w3.org/2000/svg"
                                                        className="attachment-full size-full"
                                                        width="40"
                                                        height="40"
                                                        viewBox="0 0 20 20"
                                                    >
                                                        <g>
                                                            <path
                                                                fill="currentColor"
                                                                d="M3 5a1 1 0 0 1 1-1h12a1 1 0 1 1 0 2H4a1 1 0 0 1-1-1zm0 5a1 1 0 0 1 1-1h6a1 1 0 1 1 0 2H4a1 1 0 0 1-1-1zm0 5a1 1 0 0 1 1-1h12a1 1 0 0 1 0 2H4a1 1 0 0 1-1-1z"
                                                                opacity="1"
                                                            />
                                                        </g>
                                                    </svg>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                           
                        </div>
                    </div>
                </div>
            </div>
            <div
                className={`full-block mobile-nav ${isMenuOpen ? 'open' : ''}`}
                style={{ display: 'none' }}
            >
                <div className="full-block overlay">
                    <div className="mobile-nav-inner">
                        <div className="menu-holder-inside">
                            <div className="inner-header-1">
                                <a href="/" className="logo">
                                    <img
                                        width="710"
                                        height="96"
                                        src={LOGO}
                                        className="attachment-full size-full"
                                        alt=""
                                    />
                                </a>

                                <div className="mobile-buttons">
                                    <a href="#" className="btn-menu open" onClick={toggleMenu}>
                                        <span className="menu-icon">
                                            <span></span>
                                            <span></span>
                                            <span></span>
                                            <span></span>
                                        </span>
                                    </a>
                                </div>
                            </div>

                            <Navigation hidePopup={hidePopup} />
                        </div>
                    </div>
                </div>
            </div>
        </header>
    );
};

export default Header;
